; Animated German Flag
; (C) 2026 Thomas Jentzsch

; Draws an animated German flag with varying "winds".

; Features:
; - Flag uses playfield only (all 40 pixel), except for right border
; - The animation between stripes is 15 pixel high
; - Flag is animated each frame
;     -> 15 * 40 = 600 pixel have to be updated each frame, leaving less than 8 cycles
;     per pixel outside kernel
; - Using byte instead of pixel updates, which savels a lot of time (5.x cycles/pixel)
; - Also 40 new sinus values have to be caclulated per frame too. To get the
;     effect of a flag attached to a pole, two phases, two sinus tables, two
;     phase tables, a blend table and an amplitude table have to mixed together
;     per sinus.
; - The sinus calculation is optimized heavily
;     (incl. 50% calculation in display kernel!)
; - The code uses the stack pointer as 3rd index:
;   - PHA for (also faster) stores to RAM
;   - PLA for loads from RAM
; - Left difficulty switch changes background

    processor 6502
  LIST OFF
    include vcs.h
    include tv_modes.h
    include jtz_macros.h
  LIST ON

; Cancelled IDEAs:
; x graphics:
;   x add shading (players), allow up to two local minimums
;   x use players for smoothing
; x speed:
;   x eliminate JSR (inline or use JMP instead)
;   x utilize vertical sync (not requrired)
;   x use sinus list deltas and update only changes bytes? but how? (not requrired)
;   x stop ROLs/RORs if result is 0 (set all following bytes to 0)
;   x reverse phase calculation
;   x put sinus calculations into RAM, to save 2 cycles/loop (aby vs. izy)
;     (needs 24 bytes RAM, would save 80 cycles total)
;   x put ROLs/RORs into RAM, sinus results too (cpy sinLst+n -> cpy #sinval)
;     needs ~45 bytes RAM (would save 40 * 15 = 600 cycles total)
;   x overlap sinus list with PF list (1 byte max.)

; DONEs:
; + speed:
;   + calculate 20 sinus values at once
;   + move phase additions into pointer additions (and double sinus tables)
;   + half size sinus tables (avoids page penalties, saves 0.5K ROM)
;   + use Phase0Tbl as end of loop marker (saves 3*40 cycles)
;   + use PLA in kernel, push in correct order
;   + precalculate sinus list in kernel (~40 * 144 cycles free)
; + graphics:
;   + use player for animating right border
;   + increase ANIM_H to 15
;   + vary phase speeds
; + size:
;   + swap 4+4 bits in sinus and blend tables


;===============================================================================
; A S S E M B L E R - S W I T C H E S
;===============================================================================

VERSION         = $0130
BASE_ADR        = $f000     ; 2K

ILLEGAL         = 1
DEBUG           = 1


;===============================================================================
; C O L O R - C O N S T A N T S
;===============================================================================

BLUE_COL    = BLUE_CYAN+$e
BLACK_COL   = BLACK
  IF NTSC_COL
RED_COL     = RED+$4    ; c52020 vs. ff0000 (Stella vs. official RGB values)
GOLD_COL    = BROWN+$e  ; f2c23f vs. ffcc00
  ELSE
RED_COL     = RED+$4    ; ae2c3e vs. ff0000
GOLD_COL    = BROWN+$c  ; fde060 vs. ffcc00
  ENDIF

;===============================================================================
; G A M E - C O N S T A N T S
;===============================================================================

EOR_RND_LO      = $bd   ; %10110100 ($9c, $b4, $bd, $ca, $eb, $fc)

STACK_SIZE      = 2

PF_W            = 40
NUM_STRIPES     = 3
STRIPE_H        = 59    ; here we could gain a bit CPU time :)
ANIM_H          = 15    ; vertical size of stripe animation

PHASE0_SPEED    = 3     ; initial values
PHASE1_SPEED    = 7

NUM_TMPS        = 1


;===============================================================================
; Z P - V A R I A B L E S
;===============================================================================

    SEG.U   variables
    ORG     $80

pfLst           ds ANIM_H * 6   ; values stored interleaved! (must be at $80!)
;---------------------------------------
randomLo        .byte
randomHi        .byte

sin0TblPtr      .word           ; pointer to sinus table + phase
phase0          = sin0TblPtr
sin1TblPtr      .word           ; pointer to sinus table + phase
phase1          = sin1TblPtr

phase0Speed     .byte
phase1Speed     .byte
;---------------------------------------
sinLst          ds PF_W/2       ; calculated for left or right PF half
;---------------------------------------
tmpVars         ds NUM_TMPS     ; can overlap with stack

RAM_END         = .

  LIST OFF


;===============================================================================
; M A C R O S
;===============================================================================

  LIST ON

    NEXT_PASS


;===============================================================================
; R O M - C O D E
;===============================================================================
    SEG     Bank0
    ORG     BASE_ADR, 0

;---------------------------------------------------------------
DrawScreen SUBROUTINE
;---------------------------------------------------------------
    START_TMP
.stripeCnt  .byte
    END_TMP

.waitTim
    lda     INTIM
    bne     .waitTim
;---------------------------------------------------------------
    ldy     #NUM_STRIPE_COLS-2
    sty     .stripeCnt
    ldx     #STRIPE_H * 3 + ANIM_H+1
    clc
.loopStripes                    ;           @52
; draw top lines of flag stripe:
    ldy     .stripeCnt          ; 3
    lda     StripeACol+1,y      ; 4             BLUE, BLACK, RED, YELLOW
    sta     COLUPF              ; 3
    lda     StripeACol,y        ; 4
    bit     SWCHB               ; 4
    sta     WSYNC               ; 3 = 21
;---------------------------------------
    bvc     .colorSetA          ; 2/3
    lda     StripeBCol+1,y      ; 4             BLACK, dark BLACK, RED, YELLOW
    sta     COLUPF              ; 3
    lda     StripeBCol,y        ; 4
.colorSetA                      ;
    sta     COLUBK              ; 3 = 6/16
    lda     #$ff                ; 2
    sta     PF0                 ; 3         @..21
    sta     PF1                 ; 3         @..24
    sta     PF2                 ; 3 = 11    @..27
    lda     Phase0VTbl,x        ; 4             A = 0..127|$80
    and     #~$80               ; 2
    tay                         ; 2 =  8        Y = 0..127
    lda     (sin0TblPtr),y      ; 5             A = 0..9
    ldy     Phase1VTbl,x        ; 4             Y = 0..127
    adc     (sin1TblPtr),y      ; 5             A = 0..15
    tay                         ; 2 = 16
    lda     AmpVTbl,y           ; 4
    sta     GRP0                ; 3 =  7    @..58
    dex                         ; 2
    txa                         ; 2
    ldx     #pfLst-1            ; 2             set SP to start of PF list
    txs                         ; 2
    tax                         ; 2
    ldy     Phase0VTbl,x        ; 4
    sta     WSYNC               ; 3 = 17    @65/75
LoopAnimStripe                  ;           @00
;---------------------------------------
    pla                         ; 4
    sta     PF0                 ; 3
    pla                         ; 4
    sta     PF1                 ; 3
    pla                         ; 4
    sta     PF2                 ; 3 = 21
    lda     (sin0TblPtr),y      ; 5             A = 0..9
    ldy     Phase1VTbl,x        ; 4             Y = 0..127
    adc     (sin1TblPtr),y      ; 5             A = 0..15
    tay                         ; 2 = 16
    pla                         ; 4
    sta     PF0                 ; 3         @44
    pla                         ; 4
    sta     PF1                 ; 3         @51
    pla                         ; 4
    sta     PF2                 ; 3 = 21    @58
    lda     AmpVTbl,y           ; 4
    sta     GRP0                ; 3 =  7    @65
    SLEEP   2                   ; 2
    dex                         ; 2
    ldy     Phase0VTbl,x        ; 4
    bpl     LoopAnimStripe      ; 3/2=11/10
    CHECKPAGE_LBL LoopAnimStripe, "LoopAnimStripe"
; PF: 6*7       = 42
; loop:            5
; border: 18+10 = 28
; total:          76
;---------------------------------------
    lda     #$00                ; 2
    sta     PF0                 ; 3
    sta     PF1                 ; 3
    sta     PF2                 ; 3 = 11
    txa                         ; 2
    beq     .exitKernel         ; 2/3= 4
    dec     .stripeCnt          ; 5
    tya                         ; 2
    and     #~$80               ; 2
    tay                         ; 2
    bpl     .enterLoopFixed     ; 3 = 14    @17

; draw fixed, middle lines of flag stripe:
.skipCalcC
    clc
    ldy     Phase0VTbl,x        ; 4
    bmi     .loopFixedStripeJmp
.loopFixedStripe                ;           @32
    sta     WSYNC
;---------------------------------------
.enterLoopFixed
    lda     (sin0TblPtr),y      ; 5             A = 0..9
    ldy     Phase1VTbl,x        ; 4             Y = 0..127
    adc     (sin1TblPtr),y      ; 5             A = 0..15
    tay                         ; 2 = 16
    lda     AmpVTbl,y           ; 4
    sta     GRP0                ; 3         @23
    dex                         ; 2 =  9
; calculate sinus list for left PF:
    cpx     #ANIM_H+2           ; 2
    bcc     .skipCalc           ; 2/3
    cpx     #PF_W/2+ANIM_H+2    ; 2
    bcs     .skipCalcC          ; 2/3= 8/9
; note: the three table accesses must NOT cross a page!
    ldy     Phase0Tbl-ANIM_H-1,x; 4             Y = 0..127
    lda     (sin0TblPtr),y      ; 5             A = 0..9
    ldy     Phase1Tbl-ANIM_H-1,x; 4             Y = 0..127
    adc     (sin1TblPtr),y      ; 5 = 18        A = 0..15
    ora     BlendTbl-ANIM_H-1,x ; 4             | BlendTbl[x] (0..15) << 4
    tay                         ; 2
; index = sum | (blend << 4)
    lda     AmpTbl,y            ; 4             A = amp[blend|sum] = 0..15 = wave value
    sta     sinLst-ANIM_H-2,x   ; 4 = 14    @65 store in sinLst
.skipCalc
; total: 33
    ldy     Phase0VTbl,x        ; 4
    bpl     .loopFixedStripe    ; 3/2= 7/6
.loopFixedStripeJmp
    jmp     .loopStripes        ; 3  = 3

;---------------------------------------------------------------
.exitKernel
    dex                         ;           X = $ff
    txs
    jmp     ContDrawScreen
; /DrawScreen

;---------------------------------------------------------------
Start SUBROUTINE
;---------------------------------------------------------------
    lda     #0
    tax
    cld                         ; clear BCD math bit
.clearLoop
    dex
    txs
    pha
    bne     .clearLoop

; initialize random generator:      @22
    lda     INTIM
    ora     #4
    sta     randomLo
; set high sinus table pointers once:
    lda     #>Sin0Tbl
    sta     sin0TblPtr+1
    lda     #>Sin1Tbl
    sta     sin1TblPtr+1
; initialize phase speeds:
    lda     #PHASE0_SPEED
    sta     phase0Speed
    lda     #PHASE1_SPEED
    sta     phase1Speed
; setup right border player:
    lda     #$80                ;       +8 pixel
    sta     HMP0
    SLEEP   10
    sta     RESP0               ;       @69!
    sta     WSYNC
;---------------------------------------
    sta     HMOVE
MainLoop
;---------------------------------------------------------------
VerticalBlank SUBROUTINE
;---------------------------------------------------------------
    lda     #%1110              ; each '1' bits generate a VSYNC ON line (bits 1..3)
.loopVSync
    sta     WSYNC               ; 1st '0' bit resets Vsync, 2nd '0' bit exits loop
;---------------------------------------
    sta     VSYNC
    sta     VBLANK              ; note: might be enabled too short on real hardware
    lsr
    bne     .loopVSync          ; branch until VSYNC has been reset

  IF NTSC_TIM
    lda     #40                 ; 39 * 64 = 2496
  ELSE
    lda     #77
  ENDIF
    sta     TIM64T

TIM_VAs
; *** update right PF data ***
; calc 2nd half of sinus list for right PF:
    lda     #PF_W-1+3           ; start of phase tables index (+3 due to markers)
    ldx     #sinLst+20-1        ; make stack point to end of filled sinus list part
    jsr     CalcSinus
; fill right PF:
    ldx     #<pfLst + ANIM_H*6-1; pointer to end of right PF list
    jsr     FillPF

; *** update initial values for next frame: ***
; create next random value:
    lda     randomLo            ; 3
    lsr                         ; 2
    rol     randomHi            ; 5
    bcc     .skipEor            ; 2/3
    eor     #EOR_RND_LO         ; 2
.skipEor
    sta     randomLo            ; 3
    eor     randomHi            ; 3
    bne     .skipChangePhase    ; 3/2=..23/22
; change phase speeds:
    and     #$03                ; 2
    adc     #2                  ; 2
    sta     phase0Speed         ; 3         2..6
    lda     randomHi            ; 3
    and     #$07                ; 2
    adc     #4                  ; 2
    sta     phase1Speed         ; 3 = 17    4..11
.skipChangePhase
; update phases:
    lda     phase0              ; 3
    sec                         ; 2
    sbc     phase0Speed         ; 3
    and     #$7f                ; 2
    sta     phase0              ; 3         0..127
    lda     phase1              ; 3
    sec                         ; 2
    sbc     phase1Speed         ; 3
    and     #$7f                ; 2
    sta     phase1              ; 3 = 26    0..127

    lda     #BLUE_COL           ; 2
    bit     SWCHB               ; 4
    bvc     .colorSetAP         ; 2/3
    lda     #BLACK_COL          ; 2
.colorSetAP
    sta     COLUP0              ; 3 = 10/11

TIM_VAe
; AMIM_H = 15: 2356 cycles
; /VerticalBlank

    jmp     DrawScreen

ContDrawScreen

;---------------------------------------------------------------
OverScan SUBROUTINE
;---------------------------------------------------------------

  IF NTSC_TIM
    lda     #38                 ; 37 * 64 = 2368
  ELSE
    lda     #63
  ENDIF
    sta     TIM64T

TIM_OAs

; *** update left PF data, prepare sinus list for right PF ***
; fill left PF:
    ldx     #<pfLst + ANIM_H*6-1-3; pointer to end of left PF list
    jsr     FillPF
; calc 1st half of sinus list for right PF:
    lda     #PF_W*3/4-1+2       ; start of phase tables index (+2 due to markers)
    ldx     #sinLst+10-1        ; make stack point to end of filled sinus list part
    jsr     CalcSinus

TIM_OAe ; 2268 cycles (ANIM_H = 15)

.waitTim
    lda     INTIM
    bne     .waitTim
; /OverScan

    jmp     MainLoop

;---------------------------------------------------------------
CalcSinus SUBROUTINE
;---------------------------------------------------------------
; 1. calculate sinus list:
TIM_Ss
    txs                         ; 2         make stack point to end of sinus list
    tax                         ; 2         29 or 39, end of phase tables
    clc                         ; 2
    ldy     Phase0Tbl,x         ; 4 = 10    Y = 0..127
LoopSinus
    lda     (sin0TblPtr),y      ; 5         A = 0..9
    ldy     Phase1Tbl,x         ; 4         Y = 0..127
    adc     (sin1TblPtr),y      ; 5 = 14    A = 0..15
    ora     BlendTbl,x          ; 4         | BlendTbl[x] (0..15) << 4
    tay                         ; 2
; index = sum | (blend * 16)
    lda     AmpTbl,y            ; 4         A = amp[blend|sum] = 0..15 = wave value
    pha                         ; 3 = 13    store in sinLst
; loop:
    dex                         ; 2
    ldy     Phase0Tbl,x         ; 4         Y = 0..127/255 (used as end of loop marker)
    bpl     LoopSinus           ; 3/2= 9/8
    CHECKPAGE_LBL LoopSinus, "LoopSinus"
; total: 36 * 10 = 360 cycles
    ldx     #$fd                ; 2
    txs                         ; 2 =  4
TIM_Se  ; 373 cycles
    rts                         ; 6 =  6

; 2. fill playfield lists:
FillPF
; X = make stack point to end of PF list
TIM_Fs
    ldy     #0                  ; 2
LoopRows
    txs                         ; 2 =  2
; fill PF2:
    cpy     sinLst+12           ; 3
    ror                         ; 2
    cpy     sinLst+13           ; 3
    ror                         ; 2
    cpy     sinLst+14           ; 3
    ror                         ; 2
    cpy     sinLst+15           ; 3
    ror                         ; 2
    cpy     sinLst+16           ; 3
    ror                         ; 2
    cpy     sinLst+17           ; 3
    ror                         ; 2
    cpy     sinLst+18           ; 3
    ror                         ; 2
    cpy     sinLst+19           ; 3
    ror                         ; 2
    pha                         ; 3 = 43
; fill PF1:
    cpy     sinLst+4            ; 3
    rol                         ; 2
    cpy     sinLst+5            ; 3
    rol                         ; 2
    cpy     sinLst+6            ; 3
    rol                         ; 2
    cpy     sinLst+7            ; 3
    rol                         ; 2
    cpy     sinLst+8            ; 3
    rol                         ; 2
    cpy     sinLst+9            ; 3
    rol                         ; 2
    cpy     sinLst+10           ; 3
    rol                         ; 2
    cpy     sinLst+11           ; 3
    rol                         ; 2
    pha                         ; 3 = 43
; fill PF0:
    cpy     sinLst+0            ; 3
    ror                         ; 2
    cpy     sinLst+1            ; 3
    ror                         ; 2
    cpy     sinLst+2            ; 3
    ror                         ; 2
    cpy     sinLst+3            ; 3
    ror                         ; 2
    pha                         ; 3 = 23
; loop:
    iny                         ; 2
    tsx                         ; 2         skip other PF half
    dex                         ; 2
    dex                         ; 2
    dex                         ; 2
    bmi     LoopRows            ; 3/2=13/12 PF data starts at $80
    CHECKPAGE_LBL LoopRows, "LoopRows"
; total: 124 * 15 = 1860
    ldx     #$fd                ; 2
    txs                         ; 2
TIM_Fe  ; 1865 cycles
    rts                         ; 6 = 10

    .byte   "JTZ"

;===============================================================================
; R O M - T A B L E S (Bank 0)
;===============================================================================

    ALIGN_FREE_LBL 256, "Copyright" ; avoids that BlendTbl-n,x crosses page in kernel

Copyright
    .byte   " Animated German Flag V"
    VERSION_STR
    .byte   " - (C) 2026 Thomas Jentzsch " ;
COPYRIGHT_LEN = . - Copyright

; 40 bytes  wave strength per X-position (0 = pole, 15 = full wave)
BlendTbl
; linear:
    .byte   -1
    .byte    0*16,  2*16,  2*16,  3*16,  3*16,  4*16,  4*16,  5*16
    .byte    6*16,  7*16,  8*16,  9*16, 10*16, 11*16, 12*16, 13*16
    .byte   14*16, 15*16, 15*16, 15*16
    .byte   -1
    .byte   15*16, 15*16, 15*16, 15*16
    .byte   15*16, 15*16, 15*16, 15*16, 15*16, 15*16
    .byte   -1
    .byte   15*16, 15*16
    .byte   15*16, 15*16, 15*16, 15*16, 15*16, 15*16, 15*16, 15*16
    CHECKPAGE_DATA_LBL BlendTbl, "BlendTbl"

;; expo:
;    .byte 0,0,0,0,1,1,1,2,2,3
;    .byte 3,4,4,5,6,7,8,9,10,11
;    .byte 12,13,14,15,15,15,15,15,15,15
;    .byte 15,15,15,15,15,15,15,15,15,15

;; soft:
;    .byte 0,0,1,1,2,2,3,3,4,4
;    .byte 5,5,6,6,7,7,8,8,9,9
;    .byte 10,10,11,11,12,12,13,13,14,14
;    .byte 15,15,14,14,13,13,12,12,11,11

;; strongwind:
;    .byte 0,0,0,1,1,2,2,3,4,5
;    .byte 6,7,8,9,10,11,12,13,14,15
;    .byte 15,15,14,13,12,13,14,15,15,15
;    .byte 15,15,15,15,15,15,15,15,15,15

;; windy:
;    .byte 0,0,0,0,1,1,2,2,3,4
;    .byte 5,6,7,8,9,10,11,12,13,14
;    .byte 15,15,15,15,14,13,14,15,15,15
;    .byte 15,15,15,15,15,15,15,15,15,15 ;

AmpVTbl
    .byte %01111111
    .byte %01111111
    .byte %00111111
    .byte %00111111
    .byte %00011111
    .byte %00011111
    .byte %00001111
    .byte %00001111
    .byte %00000111
    .byte %00000111
    .byte %00000011
    .byte %00000011
    .byte %00000001
    .byte %00000001
    .byte %00000000
    .byte %00000000
    CHECKPAGE_DATA_LBL AmpVTbl, "AmpVTbl"

StripeACol
    .byte   BLUE_COL, GOLD_COL, RED_COL, BLACK_COL, BLUE_COL
NUM_STRIPE_COLS = . - StripeACol
    CHECKPAGE_DATA_LBL StripeACol, "StripeACol"
StripeBCol
    .byte   BLACK_COL, GOLD_COL, RED_COL+2, BLACK_COL+2, BLACK_COL
    CHECKPAGE_DATA_LBL StripeBCol, "StripeBCol"

    ALIGN_FREE_LBL 256, "Sin0Tbl"

Sin0Tbl     ; 0..9
  REPEAT 2  ; table doubled for faster processing
    .byte   5 ; 0
    .byte   5 ; 1
    .byte   5 ; 2
    .byte   5 ; 3
    .byte   5 ; 4
    .byte   6 ; 5
    .byte   6 ; 6
    .byte   6 ; 7
    .byte   6 ; 8
    .byte   6 ; 9
    .byte   7 ; 10
    .byte   7 ; 11
    .byte   7 ; 12
    .byte   7 ; 13
    .byte   7 ; 14
    .byte   8 ; 15
    .byte   8 ; 16
    .byte   8 ; 17
    .byte   8 ; 18
    .byte   8 ; 19
    .byte   8 ; 20
    .byte   8 ; 21
    .byte   8 ; 22
    .byte   9 ; 23
    .byte   9 ; 24
    .byte   9 ; 25
    .byte   9 ; 26
    .byte   9 ; 27
    .byte   9 ; 28
    .byte   9 ; 29
    .byte   9 ; 30
    .byte   9 ; 31
    .byte   9 ; 32
    .byte   9 ; 33
    .byte   9 ; 34
    .byte   9 ; 35
    .byte   9 ; 36
    .byte   9 ; 37
    .byte   9 ; 38
    .byte   9 ; 39
    .byte   9 ; 40
    .byte   9 ; 41
    .byte   8 ; 42
    .byte   8 ; 43
    .byte   8 ; 44
    .byte   8 ; 45
    .byte   8 ; 46
    .byte   8 ; 47
    .byte   8 ; 48
    .byte   8 ; 49
    .byte   7 ; 50
    .byte   7 ; 51
    .byte   7 ; 52
    .byte   7 ; 53
    .byte   7 ; 54
    .byte   6 ; 55
    .byte   6 ; 56
    .byte   6 ; 57
    .byte   6 ; 58
    .byte   6 ; 59
    .byte   5 ; 60
    .byte   5 ; 61
    .byte   5 ; 62
    .byte   5 ; 63
    .byte   5 ; 64
    .byte   4 ; 65
    .byte   4 ; 66
    .byte   4 ; 67
    .byte   4 ; 68
    .byte   3 ; 69
    .byte   3 ; 70
    .byte   3 ; 71
    .byte   3 ; 72
    .byte   3 ; 73
    .byte   2 ; 74
    .byte   2 ; 75
    .byte   2 ; 76
    .byte   2 ; 77
    .byte   2 ; 78
    .byte   1 ; 79
    .byte   1 ; 80
    .byte   1 ; 81
    .byte   1 ; 82
    .byte   1 ; 83
    .byte   1 ; 84
    .byte   1 ; 85
    .byte   1 ; 86
    .byte   0 ; 87
    .byte   0 ; 88
    .byte   0 ; 89
    .byte   0 ; 90
    .byte   0 ; 91
    .byte   0 ; 92
    .byte   0 ; 93
    .byte   0 ; 94
    .byte   0 ; 95
    .byte   0 ; 96
    .byte   0 ; 97
    .byte   0 ; 98
    .byte   0 ; 99
    .byte   0 ; 100
    .byte   0 ; 101
    .byte   0 ; 102
    .byte   0 ; 103
    .byte   0 ; 104
    .byte   0 ; 105
    .byte   1 ; 106
    .byte   1 ; 107
    .byte   1 ; 108
    .byte   1 ; 109
    .byte   1 ; 110
    .byte   1 ; 111
    .byte   1 ; 112
    .byte   1 ; 113
    .byte   2 ; 114
    .byte   2 ; 115
    .byte   2 ; 116
    .byte   2 ; 117
    .byte   2 ; 118
    .byte   3 ; 119
    .byte   3 ; 120
    .byte   3 ; 121
    .byte   3 ; 122
    .byte   3 ; 123
    .byte   4 ; 124
    .byte   4 ; 125
    .byte   4 ; 126
    .byte   4 ; 127
  REPEND
    CHECKPAGE_DATA_LBL Sin0Tbl, "Sin0Tbl"

    ALIGN_FREE_LBL 256, "Sin1Tbl"

Sin1Tbl     ; 0..6
  REPEAT 2  ; table doubled for faster processing
    .byte   3 ; 0
    .byte   3 ; 1
    .byte   3 ; 2
    .byte   3 ; 3
    .byte   4 ; 4
    .byte   4 ; 5
    .byte   4 ; 6
    .byte   4 ; 7
    .byte   4 ; 8
    .byte   4 ; 9
    .byte   4 ; 10
    .byte   5 ; 11
    .byte   5 ; 12
    .byte   5 ; 13
    .byte   5 ; 14
    .byte   5 ; 15
    .byte   5 ; 16
    .byte   5 ; 17
    .byte   5 ; 18
    .byte   5 ; 19
    .byte   5 ; 20
    .byte   6 ; 21
    .byte   6 ; 22
    .byte   6 ; 23
    .byte   6 ; 24
    .byte   6 ; 25
    .byte   6 ; 26
    .byte   6 ; 27
    .byte   6 ; 28
    .byte   6 ; 29
    .byte   6 ; 30
    .byte   6 ; 31
    .byte   6 ; 32
    .byte   6 ; 33
    .byte   6 ; 34
    .byte   6 ; 35
    .byte   6 ; 36
    .byte   6 ; 37
    .byte   6 ; 38
    .byte   6 ; 39
    .byte   6 ; 40
    .byte   6 ; 41
    .byte   6 ; 42
    .byte   6 ; 43
    .byte   5 ; 44
    .byte   5 ; 45
    .byte   5 ; 46
    .byte   5 ; 47
    .byte   5 ; 48
    .byte   5 ; 49
    .byte   5 ; 50
    .byte   5 ; 51
    .byte   5 ; 52
    .byte   5 ; 53
    .byte   4 ; 54
    .byte   4 ; 55
    .byte   4 ; 56
    .byte   4 ; 57
    .byte   4 ; 58
    .byte   4 ; 59
    .byte   4 ; 60
    .byte   3 ; 61
    .byte   3 ; 62
    .byte   3 ; 63
    .byte   3 ; 64
    .byte   3 ; 65
    .byte   3 ; 66
    .byte   3 ; 67
    .byte   2 ; 68
    .byte   2 ; 69
    .byte   2 ; 70
    .byte   2 ; 71
    .byte   2 ; 72
    .byte   2 ; 73
    .byte   2 ; 74
    .byte   1 ; 75
    .byte   1 ; 76
    .byte   1 ; 77
    .byte   1 ; 78
    .byte   1 ; 79
    .byte   1 ; 80
    .byte   1 ; 81
    .byte   1 ; 82
    .byte   1 ; 83
    .byte   1 ; 84
    .byte   0 ; 85
    .byte   0 ; 86
    .byte   0 ; 87
    .byte   0 ; 88
    .byte   0 ; 89
    .byte   0 ; 90
    .byte   0 ; 91
    .byte   0 ; 92
    .byte   0 ; 93
    .byte   0 ; 94
    .byte   0 ; 95
    .byte   0 ; 96
    .byte   0 ; 97
    .byte   0 ; 98
    .byte   0 ; 99
    .byte   0 ; 100
    .byte   0 ; 101
    .byte   0 ; 102
    .byte   0 ; 103
    .byte   0 ; 104
    .byte   0 ; 105
    .byte   0 ; 106
    .byte   0 ; 107
    .byte   1 ; 108
    .byte   1 ; 109
    .byte   1 ; 110
    .byte   1 ; 111
    .byte   1 ; 112
    .byte   1 ; 113
    .byte   1 ; 114
    .byte   1 ; 115
    .byte   1 ; 116
    .byte   1 ; 117
    .byte   2 ; 118
    .byte   2 ; 119
    .byte   2 ; 120
    .byte   2 ; 121
    .byte   2 ; 122
    .byte   2 ; 123
    .byte   2 ; 124
    .byte   3 ; 125
    .byte   3 ; 126
    .byte   3 ; 127
  REPEND
    CHECKPAGE_DATA_LBL Sin1Tbl, "Sin1Tbl"

    ALIGN_FREE_LBL 256, "AmpTbl"

; 256 Bytes: amp[sum + blend*16] = sum * blend

AmpTbl
;; hand optimized from table below this one
;    .byte 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0          ; sum 0
;    .byte 0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1          ; sum 1
;    .byte 0,0,0,0,0,0,0,0,0,0,0,1,1,2,2,2          ; sum 2
;    .byte 0,0,0,0,0,0,0,0,0,1,1,1,2,3,3,3          ; sum 3
;    .byte 0,0,0,0,0,0,0,1,1,1,1,2,3,4,4,4          ; sum 4
;    .byte 0,0,0,0,0,0,1,1,1,1,2,3,4,5,5,5          ; sum 5
;    .byte 0,0,0,0,0,1,1,1,1,2,3,4,5,6,6,6          ; sum 6
;    .byte 0,0,0,0,1,1,1,1,2,3,4,5,6,7,7,7          ; sum 7
;    .byte 0,0,0,0,1,1,1,2,3,4,5,6,7,8,8,8          ; sum 8
;    .byte 0,0,0,1,1,1,2,3,4,5,6,7,8,9,9,9          ; sum 9
;    .byte 0,0,0,1,1,2,3,4,5,6,7,8,9,10,10+1,10     ; sum 10
;    .byte 0,0,1,1,2,3,4,5,6,7,8,9,10,11,11,11-1    ; sum 11
;    .byte 0,0,1,2,3,4,5,6,7,8,9,10,11,12,12,12-1   ; sum 12

;; table above transponed:
;; for ANIM_H = 12
;    .byte 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0   ; row 0
;    .byte 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0   ; row 1
;    .byte 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1,  0, 0, 0   ; row 2
;    .byte 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 2,  0, 0, 0   ; row 3
;    .byte 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 3,  0, 0, 0   ; row 4
;    .byte 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 3, 4,  0, 0, 0   ; row 5
;    .byte 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 3, 4, 5,  0, 0, 0   ; row 6
;    .byte 0, 0, 0, 0, 1, 1, 1, 1, 2, 3, 4, 5, 6,  0, 0, 0   ; row 7
;    .byte 0, 0, 0, 0, 1, 1, 1, 2, 3, 4, 5, 6, 7,  0, 0, 0   ; row 8
;    .byte 0, 0, 0, 1, 1, 1, 2, 3, 4, 5, 6, 7, 8,  0, 0, 0   ; row 9
;    .byte 0, 0, 0, 1, 1, 2, 3, 4, 5, 6, 7, 8, 9,  0, 0, 0   ; row 10
;    .byte 0, 0, 1, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,  0, 0, 0   ; row 11
;    .byte 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,  0, 0, 0   ; row 12
;    .byte 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,  0, 0, 0   ; row 13
;    .byte 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,11,11,12,  0, 0, 0   ; row 14
;    .byte 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,10,11,  0, 0, 0   ; row 15

; for ANIM_H = 15
    .byte  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0   ; row 0
    .byte  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1   ; row 1
    .byte  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2   ; row 2
    .byte  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 3   ; row 3
    .byte  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 2, 3, 4   ; row 4
    .byte  0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 3, 4, 5   ; row 5
    .byte  0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 2, 3, 4, 5, 6   ; row 6
    .byte  0, 0, 0, 0, 1, 1, 1, 1, 1, 2, 2, 3, 4, 5, 6, 7   ; row 7
    .byte  0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 3, 4, 5, 6, 7, 8   ; row 8
    .byte  0, 0, 0, 1, 1, 1, 1, 2, 2, 3, 4, 5, 6, 7, 8, 9   ; row 9
    .byte  0, 0, 0, 1, 1, 1, 2, 2, 3, 4, 5, 6, 7, 8, 9,10   ; row 10
    .byte  0, 0, 1, 1, 1, 2, 2, 3, 4, 5, 6, 7, 8, 9,10,11   ; row 11
    .byte  0, 0, 1, 1, 2, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12   ; row 12
    .byte  0, 1, 1, 2, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13   ; row 13
    .byte  0, 1, 2, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14   ; row 14
    .byte  0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15   ; row 15
    CHECKPAGE_DATA_LBL AmpTbl, "AmpTbl"

    ALIGN_FREE_LBL 256, "Phase0VertTbl"
Phase0VTbl
    .byte   -1              ; end of loop marker
    LIST OFF
_VAL SET -1
  REPEAT NUM_STRIPES
   REPEAT ANIM_H            ; animated loop
_VAL SET _VAL + 1
   LIST ON
    .byte   _VAL & $7f
   LIST OFF
   REPEND
_VAL SET _VAL + 1
   LIST ON
    .byte   _VAL|$80        ; end of loop marker
   LIST OFF
   REPEAT STRIPE_H-ANIM_H-2 ; fixed loop
_VAL SET _VAL + 1
   LIST ON
    .byte   _VAL & $7f
   LIST OFF
   REPEND
_VAL SET _VAL + 1
   LIST ON
    .byte   _VAL|$80        ; end of loop marker
   LIST OFF
  REPEND

  REPEAT ANIM_H
_VAL SET _VAL + 1
   LIST ON
    .byte   _VAL & $7f
   LIST OFF
  REPEND
  LIST ON

    CHECKPAGE_DATA_LBL Phase0VTbl, "Phase0VTbl"

Phase0Tbl
;    .byte   0, 4, 8, 12, 16, 20, 24, 28
;    .byte   32, 36, 40, 44, 48, 52, 56, 60
;    .byte   64, 68, 72, 76, 80, 84, 88, 92
;    .byte   96, 100, 104, 108, 112, 116, 120, 124
;    .byte   128, 132, 136, 140, 144, 148, 152, 156

  LIST OFF
_VAL SET 0
; first 20 values:
   LIST ON
    .byte   -1          ; end of loop marker
   LIST OFF
   REPEAT PF_W/2
   LIST ON
    .byte   <_VAL
   LIST OFF
_VAL SET (_VAL + 2) & $7f
   REPEND
; last 2x 10 values:
  REPEAT 2              ; left, right PF
   LIST ON
    .byte   -1          ; end of loop marker
   LIST OFF
   REPEAT PF_W/4
   LIST ON
    .byte   <_VAL
   LIST OFF
_VAL SET (_VAL + 2) & $7f
   REPEND
  REPEND

  LIST ON

;  .byte 0,2,4,6,8,10,12,14
;  .byte 16,18,20,22,24,26,28,30
;  .byte 32,34,36,38,40,42,44,46
;  .byte 48,50,52,54,56,58,60,62
;  .byte 64,66,68,70,72,74,76,78
    CHECKPAGE_DATA_LBL Phase0Tbl, "Phase0Tbl"

    ALIGN_FREE_LBL 256, "Phase1VertTbl"
Phase1VTbl:
  LIST OFF
_VAL SET 0
_DIFF SET 1
  REPEAT 192
   LIST ON
    .byte   <_VAL
   LIST OFF
_VAL SET (_VAL + _DIFF * 100 / 40) & $7f
  REPEND
  LIST ON
    CHECKPAGE_DATA_LBL Phase1VTbl, "Phase1VTbl"

Phase1Tbl
;  .byte 0,6,12,18,24,30,36,42
;  .byte 48,54,60,66,72,78,84,90
;  .byte 96,102,108,114,120,126,132,138
;  .byte 144,150,156,162,168,174,180,186
;  .byte 192,198,204,210,216,222,228,234

;  .byte 0,3,6,9,12,15,18,21
;  .byte 24,27,30,33,36,39,42,45
;  .byte 48,51,54,57,60,63,66,69
;  .byte 72,75,78,81,84,87,90,93
;  .byte 96,99,102,105,108,111,114,117

;  .byte 0,3,7,12,18,25,33,42
;  .byte 52,63,75,88,102,117,133,150
;  .byte 168,187,207,228,250,17,41,66
;  .byte 92,119,147,176,206,237,13,44
;  .byte 76,109,143,178,214,251,31,68

;    .byte   0, 1, 3, 6, 10, 15, 21, 28
;    .byte   36, 45, 55, 66, 78, 91, 105, 120
;    .byte   136, 153, 171, 190, 210, 231, 253, 20
;    .byte   44, 69, 95, 122, 150, 179, 209, 240
;    .byte   16, 49, 83, 118, 154, 191, 229, 12

  LIST OFF
_VAL SET 0
_DIFF SET 0
  LIST ON
    .byte   -1          ; end of loop marker
  LIST OFF
; first 20 values:
  REPEAT PF_W/2
   LIST ON
    .byte   <_VAL
   LIST OFF
_DIFF SET _DIFF + 1
_VAL SET (_VAL + _DIFF * 35 / 100) & $7f
  REPEND
; last 2x 10 values:
  REPEAT 2              ; left, right PF
   LIST ON
    .byte   -1          ; end of loop marker
   LIST OFF
   REPEAT PF_W/4
   LIST ON
    .byte   <_VAL
   LIST OFF
_DIFF SET _DIFF + 1
_VAL SET (_VAL + _DIFF * 35 / 100) & $7f
   REPEND
  REPEND
  LIST ON
    CHECKPAGE_DATA_LBL Phase1Tbl, "Phase1Tbl"

    ORG_FREE_LBL (. & $f800) | $7fc, "Vectors"
    .word   Start
    .word   Start


;===============================================================================
; O U T P U T
;===============================================================================

    LIST OFF
    ECHO ""
    ECHO "*** Free RAM   :", [$100 - STACK_SIZE - RAM_END]d, "bytes ***"
    ECHO "*** Free ROM   :", [FREE_TOTAL + DEBUG_BYTES]d, "bytes ***"
  IF DEBUG_BYTES
    ECHO ""
    ECHO "*** Debug bytes:", [DEBUG_BYTES]d, "bytes ***"
  ENDIF
